/*==============================================================================
MASTER FILE: Deposit Franchise Runs Replication Package
==============================================================================

Purpose:
This master file executes the complete replication for the paper "Deposit Franchise Runs"

Structure:
- Part 1: Data preparation from raw sources
- Part 2: Estimation
- Part 3: Analysis, tables, and figures

Requirements:
- Stata 17 or later

Last updated: Aug 9, 2025
==============================================================================*/
clear all
set more off // Ensure scripts run without pausing for --more--
set linesize 120 // Set line width for log file readability

// Install required packages if not already installed
foreach pkg in groups estout jdbc outreg texdoc winsor2 reghdfe ftools gtools unique distinct numdate binscatter mycolours {
    capture which `pkg'
    if _rc == 111 {
      display as text "Package `pkg' not found. Attempting to install..."
      capture ssc install `pkg'
      if _rc != 0 {
          display as error "Failed to install package `pkg'. Please install manually."
      }
      else {
          display as text "Successfully installed package `pkg'."
      }
    }
  }

/*==============================================================================
GLOBAL SETTINGS
==============================================================================*/

// Set root directory (MODIFY THIS PATH TO YOUR LOCAL SETUP)
global path_root ""

// Define subdirectories
global path_code "$path_root/code"
global path_data "$path_root/data"
global path_raw "$path_data/raw"
global path_clean "$path_data/clean"
global path_temp "$path_data/temp"
global path_output "$path_root/output"
global path_tables "$path_output/tables"
global path_figures "$path_output/figures"

// Set main call reports dataset
global raw_data_DSSW "$path_raw/call_reports.dta"

// Set log file
capture log close
capture erase "$path_output/replication_log.txt" // delete any existing log file
log using "$path_output/replication_log.txt", replace text

/*=============================================================================
PART 1: DATA PREPARATION
========================================================
======================*/
display "Starting Part 1: Data Preparation"
display "Current time: " c(current_time)

// Execute master script for data cleaning and initial processing
do "$path_code/01_data_preparation/master_data_prep.do"

/*==============================================================================
PART 2: ESTIMATION
==============================================================================*/
// set globals for the main analysis (the extension suffix is used in the 04_analysis section to differentiate between different versions of the analysis, which reuse the same code)
global ext_suffix "" // Set empty extension suffix for the main version
global runoff = 0.1 // Assumed deposit outflow rate (e.g., 10% of deposits); used in deposit franchise value calculations (2d_deposit_franchises.do)
global run_u = 1          // portion of uninsured deposits that run
global run_i = 0          // portion of insured deposits that run
global ir_suffix "" // Set empty suffix for the main version of interest rate variables
global cap_rule "" // Set empty capital requirement formula for the main version
global uninsured_cap = 1 // Uninsured deposits not capped

// Set globals for main analysis output names
global fig_deposit_beta "figure_6"
global fig_binned_scatter_dec "figure_7_dec2021" 
global fig_binned_scatter_feb "figure_7_feb2023"
global fig_scatter_annotated_dec "figure_8_dec2021"
global fig_scatter_annotated_feb "figure_8_feb2023"
global tab_valuations_a "table2_panel_a"
global tab_valuations_b "table2_panel_b"
global periods_to_plot "dec2021 feb2023" // Main analysis includes both periodsp

display "Starting Part 2: Variable Construction"
display "Current time: " c(current_time)

// Execute master script for variable construction and estimation
do "$path_code/02_estimation/master_estimation.do"

/*==============================================================================
PART 3: ANALYSIS
==============================================================================*/
// this is the main analysis
// it uses the constructed variables from Part 2 (and inherits the globals from Part 2)
display "Starting Part 3: Analysis"  
display "Current time: " c(current_time) 

// Execute master analysis script
do "$path_code/03_analysis/master_analysis.do"

/*==============================================================================
COMPLETION
==============================================================================*/
display "Replication completed successfully"
display "End time: " c(current_time)

// Clean up temporary files
local files : dir "$path_temp" files "*"
foreach file of local files {
  erase `"$path_temp/`file'"'
}

log close
